/// <reference types="cypress" />

describe('Template Content in HTML mode should', () => {
    it('Process: simple if expression', () => {
      const expected = '<div><h3>Your are using exist-db.</h3></div>'
      cy.request({
        method: 'POST',
        url: '/',
        body: {
          template: '<div>\n  [% if system:get-product-name() = "eXist" %]\n  <h3>Your are using exist-db.</h3>\n  [% else %]\n  <h3>Please update your system!</h3>\n  [% endif %]\n</div>',
          params: {},
          mode: 'html'
        },
        failOnStatusCode: false
      }).then(response => {
        expect(response.status).to.eq(200)
        expect(response.body).to.have.property('result')
        expect(response.body.result).html.to.equal(expected)
      })
    })

    it('Process: else if', () => {
      const expected = '<div><h3>We recommend an update!</h3></div>'
      cy.request({
        method: 'POST',
        url: '/',
        body: {
          template: '<div>\n  [% if $version >= 6 %]\n  <h3>Your system is up to date.</h3>\n  [% elif $version <= 6 %]\n  <h3>We recommend an update!</h3>\n  [% else %]\n  <h3>Please update your system!</h3>\n  [% endif %]\n</div>',
          params: {version: 5},
          mode: 'html'
        },
        failOnStatusCode: false
      }).then(response => {
        expect(response.status).to.eq(200)
        expect(response.body).to.have.property('result')
        expect(response.body.result).html.to.equal(expected)
      })
    })

    it('Process: for loop', () => {
      const expected = '<div><ul><li>lion</li><li>cat</li><li>dog</li><li>pig</li></ul></div>'
      cy.request({
        method: 'POST',
        url: '/',
        body: {
          template: '<div>\n  <ul>\n    [% for $animal in $animals?* %]\n      <li>[[$animal]]</li>\n    [% endfor %]\n  </ul>\n</div>',
          params: {animals: ['lion', 'cat', 'dog', 'pig']},
          mode: 'html'
        },
        failOnStatusCode: false
      }).then(response => {
        expect(response.status).to.eq(200)
        expect(response.body).to.have.property('result')
        expect(response.body.result).html.to.equal(expected)
      })
    })

    it('Process: let expression', () => {
      const expected = '<p>You are running on eXist</p>'
      cy.request({
        method: 'POST',
        url: '/',
        body: {
          template: '[% let $product = system:get-product-name() %]\n  <p>You are running on [[$product]]</p>\n[% endlet %]',
          params: {},
          mode: 'html'
        },
        failOnStatusCode: false
      }).then(response => {
        expect(response.status).to.eq(200)
        expect(response.body).to.have.property('result')
        expect(response.body.result).html.to.equal(expected)
      })
    })

    it('Process: nested for with if/else', () => {
      const expected = '<ul><li>1: odd</li><li>2: even</li><li>3: odd</li><li>4: even</li><li>5: odd</li></ul>'
      cy.request({
        method: 'POST',
        url: '/',
        body: {
          template: '<ul>\n[% for $c in 1 to 5 %]\n  [% if $c mod 2 = 0 %]\n  <li>[[ $c ]]: even</li>\n  [% else %]\n  <li>[[ $c ]]: odd</li>\n  [% endif %]\n[% endfor %]\n</ul>',
          params: {},
          mode: 'html'
        },
        failOnStatusCode: false
      }).then(response => {
        expect(response.status).to.eq(200)
        expect(response.body).to.have.property('result')
        expect(response.body.result).html.to.eq(expected)
      })
    })

    it('Process: If/else with nested for', () => {
      const expected = '<ul class="documents"><li>one</li><li>two</li></ul>'
      cy.request({
        method: 'POST',
        url: '/',
        body: {
          template: '<ul class="documents">\n  [% if empty($documents) %]\n  <li>No documents</li>\n  [% else %]\n    [% for $doc in $documents?* %]\n    <li>[[$doc]]</li>\n    [% endfor %]\n  [% endif %]\n</ul>',
          params: {documents: ['one', 'two']},
          mode: 'html'
        },
        failOnStatusCode: false
      }).then(response => {
        expect(response.status).to.eq(200)
        expect(response.body).to.have.property('result')
        expect(response.body.result).html.to.eq(expected)
      })
    })

    it('include partials', () => {
      const expected = '<div><p>The footer below is loaded from <code>pages/footer.html</code></p><footer style="border-top: 1px solid #a0a0a0; margin-top: 1rem;"><p>Generated by TEI Publisher running on eXist.</p></footer><p>© e-editiones</p></div>'
      cy.request({
        method: 'POST',
        url: '/',
        body: {
          template: '<div>\n  <p>The footer below is loaded from <code>pages/footer.html</code></p>\n  [% include "pages/footer.html" %]\n  [# We can also address blocks in the included template #]\n  [% block copyright %]\n  <p>© e-editiones</p>\n  [% endblock %]\n</div>',
          params: {app: 'TEI Publisher'},
          mode: 'html'
        },
        failOnStatusCode: false
      }).then(response => {
        expect(response.status).to.eq(200)
        expect(response.body).to.have.property('result')
        expect(response.body.result).html.to.eq(expected)
      })
    })

    it('Use imported module functions', () => {
      const expected = '<section><ul><li>Application title: My great new application</li></ul></section>'
      cy.request({
        method: 'POST',
        url: '/',
        body: {
          template: '[% import "https://tei-publisher.com/jinks/xquery/demo" as "demo" at "modules/demo.xql" %]\n\n[# call function in imported demo module #]\n<section>[[demo:sysinfo($title)]]</section>',
          params: {title: 'My great new application'},
          mode: 'html'
        },
        failOnStatusCode: false
      }).then(response => {
        expect(response.status).to.eq(200)
        expect(response.body).to.have.property('result')
        expect(response.body.result).html.to.eq(expected)
      })
    })

    it('Use frontmatter', () => {
      const expected = '<article><h1>Lorem ipsum dolor sit amet</h1><p>Consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Ut enim ad minim veniam, quis nostrud exercitation ullamco laboris nisi ut aliquip ex ea commodo consequat. Duis aute irure dolor in reprehenderit in voluptate velit esse cillum dolore eu fugiat nulla pariatur. Excepteur sint occaecat cupidatat non proident, sunt in culpa qui officia deserunt mollit anim id est laborum.</p><footer>Published by Hans.</footer></article>'
      cy.request({
        method: 'POST',
        url: '/',
        body: {
          template: '---json\n{\n  "title": "Lorem ipsum dolor sit amet",\n  "author": "Hans"\n}\n---\n<article>\n<h1>[[ $title ]]</h1>\n\n<p>Consectetur adipiscing elit, sed do eiusmod tempor incididunt ut labore et dolore magna aliqua. Ut enim ad minim veniam, quis nostrud exercitation ullamco laboris nisi ut aliquip ex ea commodo consequat. Duis aute irure dolor in reprehenderit in voluptate velit esse cillum dolore eu fugiat nulla pariatur. Excepteur sint occaecat cupidatat non proident, sunt in culpa qui officia deserunt mollit anim id est laborum.</p>\n\n<footer>Published by [[$author]].</footer>\n</article>',
          params: {author: 'Rudi'},
          mode: 'html'
        },
        failOnStatusCode: false
      }).then(response => {
        expect(response.status).to.eq(200)
        expect(response.body).to.have.property('result')
        expect(response.body.result).html.to.eq(expected)
      })
    })

    it('Inherit template with extends', () => {
      const expected = '<div><header><nav><ul><li>page.html</li><li>base.html</li><li>Extra menu item</li><li>blocks.html</li></ul></nav></header><main><article><section><div><p>Extends <code>base.html</code>, which again extends <code>page.html</code>. On each level we add menu items to the menu.</p><p>A block with a default content not referenced by a template.</p></div><p>This paragraph was imported from the parent template in <code>pages/demo/base.html</code>.</p></section></article></main><footer style="border-top: 1px solid #a0a0a0; margin-top: 1rem;"><p>Generated by TEI Publisher running on eXist.</p><p>© e-editiones</p></footer></div>'
      cy.request({
        method: 'POST',
        url: '/',
        body: {
          template: '---json\n{\n  "title": "Lorem ipsum dolor sit amet",\n  "author": "Hans",\n  "templating": {\n    "extends": "pages/base.html",\n    "use": ["pages/blocks.html"]\n  }\n}\n---\n[% template menu %]\n<li>Extra menu item</li>\n[% endtemplate %]\n[# We can also address blocks in templates included by base #]\n[% template copyright %]\n<p>© e-editiones</p>\n[% endtemplate %]\n<div>\n  <p>Extends <code>base.html</code>, which again extends <code>page.html</code>. On each level we add menu items to the menu.</p>\n  [% block foo %]\n  <p>A block with a default content not referenced by a template.</p>\n  [% endblock %]\n</div>',
          params: {app: 'TEI Publisher'},
          mode: 'html'
        },
        failOnStatusCode: false
      }).then(response => {
        expect(response.status).to.eq(200)
        expect(response.body).to.have.property('result')
        expect(response.body.result).html.to.eq(expected)
      })
    })

    it('Process other templating parameters in frontmatter', () => {
      const expected = '<div><header><nav><ul><li>page.html</li><li>base.html</li></ul></nav></header><main><article><section><h1>TEI Publisher</h1><p>TEI Publisher is a web-based application for creating and publishing TEI documents.</p><footer>Published by Heinz.</footer><p>This paragraph was imported from the parent template in <code>pages/demo/base.html</code>.</p></section></article></main><footer style="border-top: 1px solid #a0a0a0; margin-top: 1rem;"><p>Generated by  running on eXist.</p></footer></div>'
      cy.request({
        method: 'POST',
        url: '/',
        body: {
          template: '---json\n{\n  "templating": {\n    "extends": "pages/base.html",\n    "namespaces": {\n      "tei": "http://www.tei-c.org/ns/1.0"\n    },\n    "modules": {\n      "https://tei-publisher.com/jinks/xquery/demo": {\n        "prefix": "demo",\n        "at": "modules/demo.xql"\n      }\n    }\n  },\n  "author": "Heinz"\n}\n---\n\n[% let $data = demo:tei() %]\n<h1>[[ $data//tei:title/text() ]]</h1>\n\n<p>[[ $data//tei:body/tei:p/text() ]]</p>\n\n<footer>Published by [[$author]].</footer>\n[% endlet %]',
          params: {author: 'Rudi'},
          mode: 'html'
        },
        failOnStatusCode: false
      }).then(response => {
        expect(response.status).to.eq(200)
        expect(response.body).to.have.property('result')
        expect(response.body.result).html.to.eq(expected)
      })
    })

    it('Process: nested for', () => {
      cy.request({
        method: 'POST',
        url: '/',
        body: {
          template: '<div>\n  <ul>\n    [% for $person in $people?* %]\n      <li>\n        <h3>[[$person?name]], [[$person?city]]</h3>\n        [% if not($person?customer) %]\n        <p>Not a customer yet.</p>\n        [% endif %]\n        [% if count($person?phone?*) > 0 %]\n        <ul>\n        [% for $number in $person?phone?* %]\n          <li>[[ $number ]]</li>\n        [% endfor %]\n        </ul>\n        [% endif %]\n      </li>\n    [% endfor %]\n  </ul>\n</div>',
          params: {people: [
            {name: 'Rudi Rüssel', city: 'Düsseldorf', customer: true, phone: ['+4978767464', '+4915152343']},
            {name: 'Anna Szczurek', city: 'Nowy Sącz', customer: false, phone: ['+48676465']},
            {name: 'Gabi Gans', city: 'Frankfurt', customer: true}
          ]},
          mode: 'html'
        },
        failOnStatusCode: false
      }).then(response => {
        expect(response.status).to.eq(200)
        expect(response.body).to.have.property('result')
        // No expected output in fixture, so just check for successful processing
      })
    })
})